const SALT = 'WelcomeSync-v1';
const ITERATIONS = 100000;

interface EncryptedPayload {
  __enc: true;
  iv: string;
  data: string;
}

type StorageChangeCallback = (
  changes: { [key: string]: chrome.storage.StorageChange },
  areaName: string
) => void;

let cachedKey: CryptoKey | null = null;

function getExtensionId(): string {
  if (typeof chrome !== 'undefined' && chrome.runtime?.id) {
    return chrome.runtime.id;
  }
  throw new Error('Extension ID not available');
}

export async function deriveKey(extensionId?: string): Promise<CryptoKey> {
  if (cachedKey && !extensionId) {
    return cachedKey;
  }

  const id = extensionId ?? getExtensionId();

  const keyMaterial = await crypto.subtle.importKey(
    'raw',
    new TextEncoder().encode(id),
    'PBKDF2',
    false,
    ['deriveKey']
  );

  const key = await crypto.subtle.deriveKey(
    {
      name: 'PBKDF2',
      salt: new TextEncoder().encode(SALT),
      iterations: ITERATIONS,
      hash: 'SHA-256',
    },
    keyMaterial,
    { name: 'AES-GCM', length: 256 },
    false,
    ['encrypt', 'decrypt']
  );

  if (!extensionId) {
    cachedKey = key;
  }

  return key;
}

export function isEncrypted(value: unknown): value is EncryptedPayload {
  return (
    typeof value === 'object' &&
    value !== null &&
    '__enc' in value &&
    (value as EncryptedPayload).__enc === true &&
    'iv' in value &&
    'data' in value
  );
}

export async function encrypt(
  data: unknown,
  key?: CryptoKey
): Promise<EncryptedPayload> {
  const cryptoKey = key ?? (await deriveKey());
  const iv = crypto.getRandomValues(new Uint8Array(12));
  const encoded = new TextEncoder().encode(JSON.stringify(data));

  const ciphertext = await crypto.subtle.encrypt(
    { name: 'AES-GCM', iv },
    cryptoKey,
    encoded
  );

  return {
    __enc: true,
    iv: btoa(String.fromCharCode(...iv)),
    data: btoa(String.fromCharCode(...new Uint8Array(ciphertext))),
  };
}

export async function decrypt<T = unknown>(
  encrypted: unknown,
  key?: CryptoKey
): Promise<T> {
  if (!isEncrypted(encrypted)) {
    return encrypted as T;
  }

  const cryptoKey = key ?? (await deriveKey());
  const iv = Uint8Array.from(atob(encrypted.iv), c => c.charCodeAt(0));
  const ciphertext = Uint8Array.from(atob(encrypted.data), c =>
    c.charCodeAt(0)
  );

  const decrypted = await crypto.subtle.decrypt(
    { name: 'AES-GCM', iv },
    cryptoKey,
    ciphertext
  );

  return JSON.parse(new TextDecoder().decode(decrypted)) as T;
}

const listeners = new Map<StorageChangeCallback, StorageChangeCallback>();

export const encryptedStorage = {
  async get<T = Record<string, unknown>>(
    keys: string | string[] | null
  ): Promise<T> {
    return new Promise((resolve, reject) => {
      const keyArray =
        keys === null ? null : Array.isArray(keys) ? keys : [keys];

      chrome.storage.local.get(keyArray, async result => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
          return;
        }

        try {
          const decrypted: Record<string, unknown> = {};
          for (const [k, v] of Object.entries(result)) {
            decrypted[k] = await decrypt(v);
          }
          resolve(decrypted as T);
        } catch (err) {
          reject(err);
        }
      });
    });
  },

  async set(items: Record<string, unknown>): Promise<void> {
    const encrypted: Record<string, EncryptedPayload> = {};

    for (const [key, value] of Object.entries(items)) {
      encrypted[key] = await encrypt(value);
    }

    return new Promise((resolve, reject) => {
      chrome.storage.local.set(encrypted, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  },

  async remove(keys: string | string[]): Promise<void> {
    return new Promise((resolve, reject) => {
      chrome.storage.local.remove(keys, () => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          resolve();
        }
      });
    });
  },

  onChanged: {
    addListener(callback: StorageChangeCallback): void {
      const wrappedCallback: StorageChangeCallback = async (
        changes,
        areaName
      ) => {
        if (areaName !== 'local') {
          callback(changes, areaName);
          return;
        }

        const decryptedChanges: {
          [key: string]: chrome.storage.StorageChange;
        } = {};

        for (const [key, change] of Object.entries(changes)) {
          decryptedChanges[key] = {
            oldValue:
              change.oldValue !== undefined
                ? await decrypt(change.oldValue)
                : undefined,
            newValue:
              change.newValue !== undefined
                ? await decrypt(change.newValue)
                : undefined,
          };
        }

        callback(decryptedChanges, areaName);
      };

      listeners.set(callback, wrappedCallback);
      chrome.storage.onChanged.addListener(wrappedCallback);
    },

    removeListener(callback: StorageChangeCallback): void {
      const wrappedCallback = listeners.get(callback);
      if (wrappedCallback) {
        chrome.storage.onChanged.removeListener(wrappedCallback);
        listeners.delete(callback);
      }
    },
  },
};

export function clearKeyCache(): void {
  cachedKey = null;
}
