/**
 * Image Optimization Utilities for Chrome Extension
 *
 * Uses pre-generated image variants for cached images.
 * Non-cached images are returned as-is (no Supabase transforms to save costs).
 */

export interface ImageTransformOptions {
  width?: number;
  height?: number;
  resize?: 'cover' | 'contain' | 'fill';
  quality?: number;
  /** Only 'origin' is supported to keep original format. Omit for auto-optimization (WebP when browser supports). */
  format?: 'origin';
}

/**
 * Pre-generated image variants available from the CMA backend.
 * These are generated at upload time using Sharp.
 */
export type ImageVariant = 'thumb' | 'medium' | 'original';

/**
 * Map presets to pre-generated variants.
 * When a cached image is detected, we use these variants instead of Supabase transforms.
 */
const PRESET_TO_VARIANT: Record<string, ImageVariant> = {
  thumbnail: 'thumb', // 150px
  carouselThumb: 'thumb', // 100px -> use thumb (150px)
  carouselMain: 'medium', // 800px
};

/**
 * Predefined image transformation presets for different contexts.
 * Note: Supabase auto-serves WebP when browser supports it, so no format needed.
 * Using resize='contain' to scale proportionally without cropping.
 */
export const IMAGE_PRESETS = {
  // Small thumbnails (listing cards, grids)
  thumbnail: { width: 150, quality: 75, resize: 'contain' as const },

  // Medium (carousel main images, manager preview)
  carouselMain: { width: 800, quality: 85, resize: 'contain' as const },

  // Small inline thumbnails (carousel strips)
  carouselThumb: { width: 100, quality: 75, resize: 'contain' as const },
} as const;

export type ImagePresetKey = keyof typeof IMAGE_PRESETS;

/**
 * Pattern to detect cached images from our storage.
 * Format: listing-{listingId}-{12-char-hash}.{ext}
 */
const CACHED_IMAGE_PATTERN = /listing-[a-f0-9-]+-[a-f0-9]{12}\.\w+$/;

/**
 * Check if a URL is a cached image from our property-images bucket.
 */
export function isCachedImageUrl(url: string | undefined | null): boolean {
  if (!url) return false;
  return url.includes('property-images') && CACHED_IMAGE_PATTERN.test(url);
}

/**
 * Get the URL for a pre-generated image variant.
 * Returns the variant URL if the image is from our cache, otherwise returns original URL.
 *
 * @param url - The original cached image URL
 * @param variant - The variant size to retrieve ('thumb', 'medium', or 'original')
 * @returns The variant URL, or original URL if variant is 'original' or URL is not cached
 */
export function getVariantUrl(
  url: string | undefined | null,
  variant: ImageVariant
): string {
  if (!url) return '';
  if (variant === 'original') return url;
  if (!isCachedImageUrl(url)) return url;

  // Insert variant suffix before file extension
  // listing-abc-123.jpg -> listing-abc-123_thumb.jpg
  return url.replace(/(\.\w+)$/, `_${variant}$1`);
}

/**
 * Get optimized image URL using pre-generated variants for cached images.
 *
 * For cached images (from property-images bucket), uses pre-generated variants.
 * For non-cached images, returns original URL (no Supabase transforms to save costs).
 *
 * @param url - The original image URL
 * @param presetOrOptions - Either a preset name (e.g., 'thumbnail') or transform options
 * @returns Variant URL for cached images, original URL for others
 *
 * @example
 * getOptimizedImageUrl(url, 'thumbnail')
 */
export function getOptimizedImageUrl(
  url: string | undefined | null,
  presetOrOptions: ImagePresetKey | ImageTransformOptions
): string {
  if (!url) return '';

  // Only use variants when a preset name is provided (not options object)
  const presetName =
    typeof presetOrOptions === 'string' ? presetOrOptions : null;

  // For cached images, use pre-generated variants
  if (isCachedImageUrl(url) && presetName) {
    const variant = PRESET_TO_VARIANT[presetName];
    if (variant) {
      return getVariantUrl(url, variant);
    }
  }

  // For non-cached images, return original URL
  return url;
}
